

//Arduino Nano Display component for Mast mounted Antenna Beam heading display
// Uses 128x64 I2C OLED for display and software serial to receive data 
// via RS485 link,
// Version 1.0 Paul McMahon VK3DIP August 2021.
// 1.1 add soft serial input parsing 
// 1.2 add LCD and leave oled for beam animation (OLED too slow otherwise)

#include <Arduino.h>
#include <U8g2lib.h>
#include <Wire.h>
#include <SoftwareSerial.h>
#include <LiquidCrystal.h>


// 128x64 OLED using U8G2 library
U8G2_SSD1306_128X64_NONAME_1_HW_I2C u8g2(U8G2_R0, /* reset=*/ U8X8_PIN_NONE); // no rotation

// Soft Serial object used for receive only from masthead sensor.
SoftwareSerial Compass(2, 3);  // RX, TX

// LCD object using 4bit interface.
LiquidCrystal lcd(8,9,4,5,6,7); // (RS,E,D4,D5,D6,D7)

const float R2D = 57.2958; // approx 180/PI for radian to degrees * or viceversa /
const float TwoPI = 6.28319; // approx 2 * PI for radians in a circle /
float headingRads; // beam heading relative to North (Y axis, up, 0/360)  Theta in radians.
int headingDegs; // beam heading relative to North (Y axis, up, 0/360)  Theta in degrees.
int oldheadingDegs; // old beam heading

// data sentence is in format "<X:nnnnn,Y:nnnnn,Z:nnnnn,>crlf where nnnnn is ascii number 0-65535 representing INT, if MSB high = neg 2s Complement
const byte numChars = 32;   // max in one sentence is 29 avg 20
char receivedChars[numChars]; // buffer for received sentence

    // variables to hold the recieved  data
int Xval = 0;
int Yval = 0;
int Zval = 0;

boolean newData = false;


void setup()
{
  Serial.begin(19200); // Serial back to PC @19200
  u8g2.begin();  //Initiate Oled display
  lcd.begin(16,2); // Initiate LCD to 16x2 16char by 2 lines
  Compass.begin(1200); // Softserial receive from masthead @1200

oldheadingDegs = 0;
 lcd.print("VK3DIP 2021 V1.2");
  // set the cursor to column 0, line 1
  // (note: line 1 is the second row, since counting begins with 0):
lcd.setCursor(0, 1);
          lcd.print("AZIMUTH  = ");
}

void loop()
{

   recvWithStartEndMarkers(); // process data from compass
    if (newData == true) {  // newdata ==true -> we have a new complete sentence
       
        echoData(); // send copy to PC serial
       
       getXYZ(); // extract and convert to the three ints Xval, Yval, Zval
       
       
       headingRads = atan2(Yval, Xval); // calculate heading from x and y
       if(headingRads < 0) {
        headingRads += TwoPI; // Correct for when signs are reversed.
       }
      headingDegs = headingRads * R2D ;// convert to degrees
  
       if(headingDegs != oldheadingDegs) {  // dont do anything unless we have changed
          oldheadingDegs = headingDegs;        
          lcd.setCursor(11, 1);
          lcd.print(headingDegs);
    //      lcd.print((char)178);     // alternate degree symbol
          lcd.print((char)223);     // degs symbol
          lcd.print("   ");     // blank out any remenants of last bearing 
           // draw screen
          u8g2.firstPage();
           do {
               drawScreen();
              } while ( u8g2.nextPage() );
       }
       
      
       newData = false;
        
    }

 
} 

// Subs and Functions
//============

void recvWithStartEndMarkers() {
// if characters are available from compass process them looking for 
// start(<)  and end(>)  put things between into string receivedChars, discard markers and the crlf
  
    static boolean recvInProgress = false;
    static byte ndx = 0;
    char startMarker = '<';
    char endMarker = '>';
    char rc;

    while (Compass.available() > 0 && newData == false) 
    {
        rc = Compass.read();
        if (rc == startMarker) 
        {
            recvInProgress = true;
            ndx=0;
        }
      
        else if (recvInProgress == true)
        {
            if (rc != endMarker)
            {
                receivedChars[ndx] = rc;
                ndx++;
                if (ndx >= numChars)
                {
                    ndx = numChars - 1;
                }
            }
            else {
                receivedChars[ndx] = '\0'; // terminate the string
                recvInProgress = false;
                ndx = 0;
                newData = true;
            }
        }

       // otherwise just ignor it 
    }
}

//============

void getXYZ() {      // retrieve the X,Y, and Z values from the received chars
// Values for X then Y then Z integers in ASCII from Compass in that order each one  with
// start(:)  and end(,)  use ATOI on each one assume ATOI will handle 2's Complement as necessary.
// data sentence is in format "X:nnnnn,Y:nnnnn,Z:nnnnn," where nnnnn is ascii number 0-65535 representing INT, if MSB high = neg 2s Complement  
 boolean  inNum = false; // true is getting a number , false is looking for one
    byte ndx = 0;
    byte nval = 0;  // Count of which value we are getting  0 = working on X, 1 = Y, 2 = Z, 3 = all done
    char startMarker = ':';
    char endMarker = ',';
    char rc;
    char tempChars[6];        // temporary array for use when parsing upto 5 digits plus null
    byte tcndx = 0; // index into tempchar
    
    
    while (nval < 3 ) { // keep working till we have all three
        rc = receivedChars[ndx]; // get next char
        ndx ++; // move the pointer ready for next time
        if (ndx >= numChars){
          nval = 3; //OOPS we have got to the end of the buffer
        }
        
      if (inNum == false) 
      {
        // looking for start
        if(rc != startMarker)
        {
          // keep looking
        }
        else
        {// it is the start marker
       inNum = true;
        }
      }  
      else 
      {
        //inNum = true and we are in a number
        if(rc != endMarker)
         {
           // we are in a number and it is not the end just collect the digit
          tempChars[tcndx] = rc;
          tcndx ++;
         }
        else 
        { 
          // it is the end marker so number is finished 
          tempChars[tcndx] = '\0'; // terminate the string
          // get the current value
          switch (nval) {
          case 0: 
             //Its X
             Xval = atoi(tempChars);     // convert this part to an integer
             break;
          case 1:
             //Its Y
            Yval = atoi(tempChars);     // convert this part to an integer
            break;
         case 2:
            //Its Z
            Zval = atoi(tempChars);     // convert this part to an integer
            break;
          }  
          inNum = false;        
          nval ++; // now looking for the next number
          tcndx = 0; //set index back to 0 for next number
        }
      }
    } 
}


//============

void echoData() {
    // assume only called when receivedChars contains a null terminated string as received from Compass 
    Serial.println(receivedChars);
  //   Serial.print("x = ");
  //Serial.println(Xval);
  //Serial.print("y = ");
 // Serial.println(Yval);
 // Serial.print("z = ");
 // Serial.println(Zval);
 //   Serial.print("Bearing = ");
  Serial.println(headingDegs); 
}

void drawScreen(){
  const int crad = 31; // = R radius of compass face
  const int cx = 64; //center x
  const int cy = 32; //center y
  
//float Theta = heading/R2D;    // convert to radians
  float SinT = sin(headingRads);
  float CosT = cos(headingRads);  //Saved sin and cos Theta (heading)


  int rx,ry, ax,ay,fx,fy,bx,by ; // coords of rim , arrow, front, and back of beam drawing and arrow
  int X0,Y0,X1,Y1 ; // Coords for lines etc.
  int delX, delY; // differences for X and Y which change with R
 
u8g2.setDrawColor(1); 

// get approx rim co-ords
  
  delX = crad*SinT;
  delY = crad*CosT;
  
  rx = cx + delX;
  ry = cy - delY;

// assume arrow is about 3/4 of way to rim

  ax = cx + 0.75 * delX;
  ay = cy + 0.75 * delY;
 
// get co-ords of boom parts

 // Assume Boom is R long centered on center, ie half way to each rim
 // get approx other ends of boom
   delX = delX/2; // strictly R/2 * SinT
  delY = delY/2; // strictly R/2 * CosT
 
  fx = cx + delX;
  fy = cy - delY;
  bx = cx - delX;
  by = cy  + delY;
  //draw boom from -R/2 to R/2
 
  u8g2.drawLine(fx, fy, bx, by);

  //draw reflector at end of boom 
  // Reflector is R long so can use existing deltas
 
 X1 = bx + delY;
 Y1 = by + delX;
 X0 = bx - delY;
 Y0 = by - delX;
 u8g2.drawLine(X0, Y0, X1, Y1);

  //draw driver in center 
  // Driver is R/1.1  long so can use existing deltas /1.1
  delX = delX/1.1; // strictly R/2.2 * SinT
  delY = delY/1.1; // strictly R/2.2 * CosT
 X0 = cx + delY;
 Y0 = cy + delX;
 X1 = cx - delY;
 Y1 = cy - delX;
 u8g2.drawLine(X0, Y0, X1, Y1);

 
 // director and arrow is 10 percent shorter  so can use existing deltas /1.1
  delX = delX/1.1; // strictly R/2.4 * SinT
  delY = delY/1.1; // strictly R/2.4 * CosT
 X0 = fx + delY;
 Y0 = fy + delX;
 X1 = fx - delY;
 Y1 = fy - delX;


// Draw Arrow
 // Draw filled then cutout section
// Draw filled
 u8g2.drawTriangle(rx, ry, X0, Y0, X1, Y1); 
 // now cuttout a section
  u8g2.setDrawColor(0);  // draw with black
  u8g2.drawTriangle((rx + fx)/2, (ry+fy)/2, X0, Y0, X1, Y1);  

 // Back to normal
 u8g2.setDrawColor(1); 
 //draw director at front 
 
 u8g2.drawLine(X0, Y0, X1, Y1);

 // Draw outer and center circle
  u8g2.drawCircle(cx, cy, crad, U8G2_DRAW_ALL);
  u8g2.drawCircle(cx, cy, 2, U8G2_DRAW_ALL);

// Draw tick marks at each Compass point
//u8g2.drawLine(cx, cy-(crad-2), cx, cy-(crad +2)); // North tick mark
//u8g2.drawLine(cx, cy+(crad-2), cx, cy+(crad +2)); // South tick mark
//u8g2.drawLine(cx-(crad-2), cy, cx-(crad+2), cy); // West tick mark
//u8g2.drawLine(cx+(crad-2), cy, cx+(crad+2), cy); // East tick mark
  //u8g2.setFont(u8g_font_unifont);
 // u8g2.setFont(u8g2_font_profont12_tf); //8 pixel font

 // u8g2.setCursor(10, 20);
 // u8g2.print("Heading: ");
// Label the Compass Directions 
//  u8g2.setCursor(cx-2, cy-(crad+1)); // need to offset each Label by a bit. - See u8g2.print() function
//  u8g2.print("N");
//  u8g2.setCursor(cx-2, cy+(crad+10)); // 2 + font size of 8 = 10
//  u8g2.print("S");
//  u8g2.setCursor(cx+(crad+4), cy+4);
//  u8g2.print("E");
//  u8g2.setCursor(cx-(crad+8), cy+4);
//  u8g2.print("W");

// Display the actual bearing in a larger font
 // u8g2.setCursor(5, 50);
//  u8g2.setFont(u8g2_font_timR24_tf); // separate font for Bearing
//  u8g2.print(headingDegs);
  
// u8g2.setFont(u8g2_font_5x7_tr);
 // u8g2.setCursor(10, 63);
//  u8g2.print("VK3DIP V1.1");
}
